package eu.usrv.yamcore.auxiliary;


import javax.annotation.Nullable;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;


public class Placeable<T> implements Supplier<T>
{
  @Nullable
  protected T value;

  public static <T> Placeable<T> make() {
    return new Placeable<>();
  }

  @Override
  public T get()
  {
    if( value == null )
    {
      throw new IllegalStateException( "Value hasn't been loaded yet." );
    }

    return value;
  }

  public void place( T value )
  {
    if( this.value != null )
    {
      throw new IllegalStateException( "Value has been already loaded!" );
    }

    this.value = value;
  }

  /**
   * Returns {@code true} if there is a mod object present, otherwise {@code false}.
   *
   * @return {@code true} if there is a mod object present, otherwise {@code false}
   */
  public boolean isLoaded()
  {
    return this.value != null;
  }

  /**
   * If a mod object is present, invoke the specified consumer with the object,
   * otherwise do nothing.
   *
   * @param consumer block to be executed if a mod object is present
   * @throws NullPointerException if mod object is present and {@code consumer} is
   *                              null
   */
  public void ifLoaded( Consumer<? super T> consumer )
  {
    if( isLoaded() )
      consumer.accept( get() );
  }

  /**
   * If a mod object is present, apply the provided mapping function to it,
   * and if the result is non-null, return an {@code Optional} describing the
   * result.  Otherwise return an empty {@code Optional}.
   *
   * @param <U>    The type of the result of the mapping function
   * @param mapper a mapping function to apply to the mod object, if present
   * @return an {@code Optional} describing the result of applying a mapping
   * function to the mod object of this {@code Promised}, if a mod object is present,
   * otherwise an empty {@code Optional}
   * @throws NullPointerException if the mapping function is null
   * @apiNote This method supports post-processing on optional values, without
   * the need to explicitly check for a return status.
   */
  public <U> Optional<U> map( Function<? super T, ? extends U> mapper )
  {
    Objects.requireNonNull( mapper );
    if( !isLoaded() )
      return Optional.empty();
    else
    {
      return Optional.ofNullable( mapper.apply( get() ) );
    }
  }

  /**
   * If a value is present, apply the provided {@code Optional}-bearing
   * mapping function to it, return that result, otherwise return an empty
   * {@code Optional}.  This method is similar to {@link #map(Function)},
   * but the provided mapper is one whose result is already an {@code Optional},
   * and if invoked, {@code flatMap} does not wrap it with an additional
   * {@code Optional}.
   *
   * @param <U>    The type parameter to the {@code Optional} returned by
   * @param mapper a mapping function to apply to the mod object, if present
   *               the mapping function
   * @return the result of applying an {@code Optional}-bearing mapping
   * function to the value of this {@code Optional}, if a value is present,
   * otherwise an empty {@code Optional}
   * @throws NullPointerException if the mapping function is null or returns
   *                              a null result
   */
  public <U> Optional<U> flatMap( Function<? super T, Optional<U>> mapper )
  {
    Objects.requireNonNull( mapper );
    if( !isLoaded() )
      return Optional.empty();
    else
    {
      return Objects.requireNonNull( mapper.apply( get() ) );
    }
  }

  /**
   * If a mod object is present, lazily apply the provided mapping function to it,
   * returning a supplier for the transformed result. If this object is empty, or the
   * mapping function returns {@code null}, the supplier will return {@code null}.
   *
   * @param <U>    The type of the result of the mapping function
   * @param mapper A mapping function to apply to the mod object, if present
   * @return A {@code Supplier} lazily providing the result of applying a mapping
   * function to the mod object of this {@code Promised}, if a mod object is present,
   * otherwise a supplier returning {@code null}
   * @throws NullPointerException if the mapping function is {@code null}
   * @apiNote This method supports post-processing on optional values, without
   * the need to explicitly check for a return status.
   */
  public <U> Supplier<U> lazyMap( Function<? super T, ? extends U> mapper )
  {
    Objects.requireNonNull( mapper );
    return () -> isLoaded() ? mapper.apply( get() ) : null;
  }

  /**
   * Return the mod object if present, otherwise return {@code other}.
   *
   * @param other the mod object to be returned if there is no mod object present, may
   *              be null
   * @return the mod object, if present, otherwise {@code other}
   */
  public T orElse( T other )
  {
    return isLoaded() ? get() : other;
  }

  /**
   * Return the mod object if present, otherwise invoke {@code other} and return
   * the result of that invocation.
   *
   * @param other a {@code Supplier} whose result is returned if no mod object
   *              is present
   * @return the mod object if present otherwise the result of {@code other.get()}
   * @throws NullPointerException if mod object is not present and {@code other} is
   *                              null
   */
  public T orElseGet( Supplier<? extends T> other )
  {
    return isLoaded() ? get() : other.get();
  }

  /**
   * Return the contained mod object, if present, otherwise throw an exception
   * to be created by the provided supplier.
   *
   * @param <X>               Type of the exception to be thrown
   * @param exceptionSupplier The supplier which will return the exception to
   *                          be thrown
   * @return the present mod object
   * @throws X                    if there is no mod object present
   * @throws NullPointerException if no mod object is present and
   *                              {@code exceptionSupplier} is null
   * @apiNote A method reference to the exception constructor with an empty
   * argument list can be used as the supplier. For example,
   * {@code IllegalStateException::new}
   */
  public <X extends Throwable> T orElseThrow( Supplier<? extends X> exceptionSupplier ) throws X
  {
    if( isLoaded() )
    {
      return get();
    }
    else
    {
      throw exceptionSupplier.get();
    }
  }
}